extensions [profiler]
 
;change repeat statement
globals 
[
  ;; environmental globals
  Q10
  theta
  surface-light   ; Set to PAR
  par             ; Photosynthetically Active Radiation
  day
  hour
  temp
  ;; hydrologic variables
  hydro-group 
  hydro-changed
  max-timestep
  patch-length

;; all biotic variables follow the same naming conventions and are in gwr/unit biomass **need refer
;; ; max- max density; e- excretion->DOC; r- respiration; s- senecence(natural mortality); -egestion ->POC
 ; j - stock 
 ; i - prey stock
 ; Cj =  PIij * THETAj * Xj * Fij * Fjj
; exampple: set waterdecomp-pred-doc (pref-waterdecomp-doc * max-waterdecomp * waterdecomp
;           * waterdecomp-space-limitation * waterdecomp-doc-prey-limitation)
 
 ;; Ai - prey density threshold max - labeled from consumer to prey
 ;; Gi - prey density threshold min - labeled from consumer to prey
 ;; Aj - space limitation max
 ;; Gj - space limitation min

; k - half saturation constant ?
; this is used in turbidity for hydrological function
;  k-inorganic ***unused

;;  DOC (dissolved organic carbon) variables  ;;

;;  POC (particulate organic carbon) variables  ;;
  k-POC ;used update-patches and 
  
;;  detritus variables  ;;

;;  phytoplankton variables  ;;
  k-phyto
  Aj-phyto           Gj-phyto
  
;;  periphyton variables (set as 10% macro per cell -* between eq3,4 source) ;;
  max-peri        e-peri        r-peri        s-peri
  Ai-Peri-DOC                 Gi-Peri-DOC                   pref-Peri-DOC 
  Ai-Peri-POC                 Gi-Peri-POC                   pref-Peri-POC
  peri-phys-loss
  Aj-peri Gj-peri
  
;;  macrophytes variables  ;;
  k-macro

;;  herbivores variables  ;;
  max-herbivore   e-herbivore   r-herbivore   s-herbivore   herbivore-egestion
  k-herbivore
  Ai-herbivore-phyto          Gi-herbivore-phyto            pref-herbivore-phyto        
  Ai-herbivore-waterdecomp    Gi-herbivore-waterdecomp      pref-herbivore-waterdecomp 
  Ai-herbivore-peri           Gi-herbivore-peri             pref-herbivore-peri
  Aj-herbivore                Gj-herbivore       

;;  decomposers in the water column variables  ;;
  max-waterdecomp e-waterdecomp r-waterdecomp s-waterdecomp
  Ai-waterdecomp-DOC          Gi-waterdecomp-DOC            pref-waterdecomp-DOC        
  Ai-waterdecomp-POC          Gi-waterdecomp-POC            pref-waterdecomp-POC
  Aj-waterdecomp              Gj-waterdecomp      
  
;;  decomposers in the sediment variables  ;;
  max-seddecomp   e-seddecomp   r-seddecomp   s-seddecomp
  Ai-seddecomp-detritus       Gi-seddecomp-detritus         pref-seddecomp-detritus
  Aj-seddecomp                Gj-seddecomp        
    
;;  consumers in the water column variables  ;;
  max-consum      e-consum      r-consum      s-consum          consum-egestion
  Ai-consum-herbivore         Gi-consum-herbivore           pref-consum-herbivore        
  Ai-consum-sedconsumer       Gi-consum-sedconsumer         pref-consum-sedconsumer 
  Aj-consum                   Gj-consum   
  
;;  consumers in the sediments variables  ;;
  max-sedconsumer e-sedconsumer r-sedconsumer  s-sedconsumer 
  sedconsumer-egestion-seddecomp sedconsumer-egestion-detritus
  Ai-sedconsumer-seddecomp    Gi-sedconsumer-seddecomp      pref-sedconsumer-seddecomp  
  Ai-sedconsumer-peri         Gi-sedconsumer-peri           pref-sedconsumer-peri       
  Ai-sedconsumer-detritus     Gi-sedconsumer-detritus       pref-sedconsumer-detritus
  Aj-sedconsumer              Gj-sedconsumer              
]


patches-own
[
  px-vector
  py-vector
  
  waterdecomp-doc-prey-limitation
  waterdecomp-poc-prey-limitation
  peri-doc-prey-limitation
  peri-poc-prey-limitation
  seddecomp-detritus-prey-limitation
  herbivore-phyto-prey-limitation
  herbivore-waterdecomp-prey-limitation
  herbivore-peri-prey-limitation
  sedconsumer-seddecomp-prey-limitation
  sedconsumer-peri-prey-limitation
  sedconsumer-detritus-prey-limitation
  consum-herbivore-prey-limitation
  consum-sedconsumer-prey-limitation
  
  peri-space-limitation           
  waterdecomp-space-limitation    
  seddecomp-space-limitation    
  herbivore-space-limitation    
  sedconsumer-space-limitation   
  consum-space-limitation      
  
  assimilation
  detritus DOC POC waterdecomp seddecomp macro phyto herbivore sedconsumer peri consum
  bottom-light
  consumer
  consum-consumption
  consum-ingest-herbivore
  consum-pred-herbivore
  consum-ingest-sedconsumer
  consum-pred-sedconsumer
  consum-excretion
  consum-sda
  consum-senescence
  consum-respiration
  consum-growth
  current-depth
  depth
  depth-in-ft
  detritus-growth
  detritus-POC-transfer
  seddecomp-pred-detritus
  sedconsumer-pred-detritus
  direction
  DOC-growth
  DOC-pred
  egestion
  excretion
  excretion-phyto
  flocculation
  gross-photo
  gross-photo-macro
  gross-photo-phyto
  growth-herbivore
  growth-detritus
  ;growth-DOC
  growth-macro
  ;growth-POC
  growth-sedconsumer
  growth-phyto
  growth-waterdecomp
  herbivore-consumption
  herbivore-ingest-peri
  herbivore-pred-peri
  herbivore-ingest-phyto
  herbivore-pred-phyto
  herbivore-ingest-waterdecomp
  herbivore-pred-waterdecomp
  herbivore-excretion
  herbivore-sda
  herbivore-senescence
  herbivore-respiration
  herbivore-growth
  K
  large-death
  light
  light-k
  macro-death
  macro-exudation
  micro-death
  output
  phyto-maximum-growth-rate 
  phyto-pred
  POC-detritus-transfer
  POC-growth
  POC-pred
  phyto-density
  peri-ingest-doc
  peri-pred-doc
  peri-ingest-poc
  peri-pred-poc
  peri-respiration
  peri-excretion
  peri-senescence
  senescence
  scouring
  small-death
  respiration
  respiration-macro
  respiration-phyto
  scouring-macro
  sedconsumer-ingest-peri
  sedconsumer-pred-peri
  senescence-macro
  senescence-phyto
  sedconsumer-consumption
  sedconsumer-ingest-detritus
  sedconsumer-ingest-seddecomp
  sedconsumer-pred-seddecomp
  sedconsumer-excretion
  sedconsumer-egestion
  sedconsumer-senescence
  sedconsumer-respiration
  sedconsumer-growth
  seddecomp-consumption
  seddecomp-ingest-detritus
  seddecomp-excretion
  seddecomp-growth
  seddcomp-ingest-peri
  seddecomp-pred-peri
  seddecomp-respiration
  seddecomp-senescence
  velocity
  velocity-in-f/s
  velpoc
  waterdecomp-consumption
  waterdecomp-ingest-doc
  waterdecomp-sda
  waterdecomp-excretion
  waterdecomp-ingest-poc
  waterdecomp-pred-doc
  waterdecomp-pred-poc
  waterdecomp-respiration
  waterdecomp-senescence
  turbidity
  tmp
  
  ; USED TO FLOW CARBON
  max-area
  cornerpatch
  tbpatch
  rlpatch

  origDOC
  origPOC
  origPHYTO
  origWATERDECOMP
  tbMvd
  cMvd
  rlMvd
  
  px
  py
]

;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;;; BEGIN SETUP FUNCTION
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;

to setup
  ; order is important here...
  ;; (for this model to work with NetLogo's new plotting features,
  ;; __clear-all-and-reset-ticks should be replaced with clear-all at
  ;; the beginning of your setup procedure and reset-ticks at the end
  ;; of the procedure.)
  __clear-all-and-reset-ticks
  setup-parameters
  setup-environmentals
  setup-stocks
  setup-output
  ask patches with [ depth <= 0 ] [ set pcolor black ]
end

to profile
                         ;; set up the model
  profiler:start         ;; start profiling
  repeat 1 [ go ]        ;; run something you want to measure
  profiler:stop          ;; stop profiling
  print profiler:report  ;; view the results
  profiler:reset         ;; clear the data
end

;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;;; BEGIN GO FUNCTION
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
to go
  
  set hour ticks mod 24
  set day floor (ticks / 24)

  update-environmentals
  ask patches with [depth > 0 and output != 1] 
  [
    update-patches 
    go-macro 
    go-phyto 
    go-herbivore  
    go-waterdecomp 
    go-seddecomp 
    go-sedconsumer
    go-consum
    go-DOC 
    go-POC 
    go-detritus 
   
    pred-phyto
    pred-herbivore
    pred-seddecomp
    pred-waterdecomp
    pred-sedconsumer
    pred-detritus
    pred-DOC
    pred-POC
    pred-consum 
  ]
   
  let steps-per-hour  3600 / max-timestep  ;(must manually change the number of times it repeats. The equation is (3600 / max-timestep)
  repeat steps-per-hour 
  [ 
    set hour hour + (1 / steps-per-hour)
    ask patches with [depth > 0 and output != 1] [ flow-carbon ]
  ]
  ; this is the hydrological flow routine, which runs at a different timestep
  ; *need patch selection - currently random - should this be algorithmic?    

    
  tick
  update-color
  do-plots
  ask patches with [pycor = 362 and pxcor > 144 and pxcor < 154]
    [ input ]
end


;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;;;;;;; END setup & go ROUTINES
;;;;;;; All called functions follow
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;



;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;;; BEGIN FUNCTIONS in SETUP ROTUTINE
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;

;
to setup-environmentals
  file-close-all
  file-open "data/par.txt"
  set par file-read
  file-close
  choose-hydro-map
  update-hydro-map
end


; function - sets initial values of all global variables
to setup-parameters
  set patch-length 97.57544757 * 0.3048 ; convert ft to m from the SMS to png scaling conversion
  set max-timestep 20 ; in seconds - this can be max patch-length / max-velocity, figure out later
  
  set max-waterdecomp 6.26 / 24  set e-waterdecomp .173 / 24  set r-waterdecomp .6 / 24 set s-waterdecomp .05 / 24 
  set max-seddecomp   6.26 / 24  set e-seddecomp .173 / 24    set r-seddecomp .6 / 24   set s-seddecomp  .05 / 24  
  set max-herbivore  1 / 24      set e-herbivore  .4 / 24    set r-herbivore .08 / 24  set s-herbivore .01 / 24     set herbivore-egestion .2
  set max-sedconsumer .55 / 24   set e-sedconsumer  .01 / 24  set r-sedconsumer .04 / 24 set s-sedconsumer .01 / 24  set sedconsumer-egestion-seddecomp .35 set sedconsumer-egestion-detritus .9
  set max-consum  .125 / 24      set e-consum  .0032 / 24     set r-consum .0125 / 24   set s-consum  .002 / 24      set consum-egestion .2
  ; max gwr/unit biomass; e- excretion->DOC; r- respiration; s- senecence(natural mortality); -egestion ->POC
  
  
  set Ai-waterdecomp-DOC  30      set Gi-waterdecomp-DOC   .05         set pref-waterdecomp-DOC .5
  set Ai-waterdecomp-POC  30      set Gi-waterdecomp-POC   .05         set pref-waterdecomp-POC .5
  set Ai-Peri-DOC       30        set Gi-Peri-DOC     .05              set pref-Peri-DOC .5
  set Ai-Peri-POC       30        set Gi-Peri-POC     .05              set pref-Peri-POC .5
  set Ai-seddecomp-detritus  .6   set Gi-seddecomp-detritus .005       set pref-seddecomp-detritus 1
  set Ai-herbivore-phyto  20       set Gi-herbivore-phyto  .01          set pref-herbivore-phyto .7
  set Ai-herbivore-waterdecomp  3 set Gi-herbivore-waterdecomp .01     set pref-herbivore-waterdecomp .15
  set Ai-herbivore-peri      3    set Gi-herbivore-peri    .01         set pref-herbivore-peri .15
  set Ai-sedconsumer-seddecomp 3  set Gi-sedconsumer-seddecomp  .02    set pref-sedconsumer-seddecomp .5
  set Ai-sedconsumer-peri   2     set Gi-sedconsumer-peri     .02      set pref-sedconsumer-peri .4
  set Ai-sedconsumer-detritus 3.5 set Gi-sedconsumer-detritus  .02     set pref-sedconsumer-detritus .1
  set Ai-consum-herbivore  3.5    set Gi-consum-herbivore   .025       set pref-consum-herbivore .7
  set Ai-consum-sedconsumer  4    set Gi-consum-sedconsumer  .04       set pref-consum-sedconsumer .3
  ; above variables are ripe for sensitivity analysis
  ; ben will send reference
  ; prey parameters - Ai and Gi, coded 1 -> 0
     
  set Aj-phyto    10       set  Gj-phyto 100
  set Aj-waterdecomp 1.2   set  Gj-waterdecomp 20     
    ;peri updated every timestep based on macro density        
  ;set Aj-seddecomp  .2     set  Gj-seddecomp  120     ; need river values 
  set Aj-herbivore  2.4    set  Gj-herbivore  108      
  set Aj-sedconsumer  2    set  Gj-sedconsumer  20    
  set Aj-consum   .65      set  Gj-consum    6.5
  ;;density dependent crowding effects on linear scale - capped at Aj and Gj - coded 1->0  
   
  set k-macro .4
  set k-phyto .4
  set k-herbivore .2
  set k-POC .2
    
  set theta 1.072
  ;let base-temp 19.7 ; base temp for nominal growth
  ;set Q10 theta ^ (temp - base-temp)
end


; function - sets initial stocks of species, wich are patch variables
;      *need measures and relations

to setup-stocks
ask patches with [ depth > 0 ]
[
 set macro 1
 set phyto 10
 set waterdecomp 10
 set seddecomp 1
 set herbivore 1
 set sedconsumer 1
 set DOC 10
 set POC 10
 set detritus 1
 set consum .1
]

end
 
; function - sets output vari of pathces
;      *need what does this do?

to setup-output
  ask patches [if pycor = 100 
    [set output 1]]
end

;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;;; END FUNCTIONS in SETUP ROTUTINE
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;



;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;;; BEGIN FUNCTIONS in GO ROTUTINE
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;

;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
; function - this sets the hydro-group varaible from the discharge hydrograph imported 
;       in the setup-environmentals routine

to choose-hydro-map
  set hydro-changed 0
  let old-hydro hydro-group
  set hydro-group "88k" 
      if old-hydro != hydro-group [ set hydro-changed 1]
end

to update-hydro-map
  let v-maps (word "data/velocity" hydro-group ".png")
  let d-maps (word "data/depth" hydro-group ".png")
  let vct-maps (word "data/vector" hydro-group ".txt")
     
     import-pcolors-rgb d-maps
     ask patches with [pcolor != 0] 
     [ 
       set depth-in-ft ( (255 - item 1 pcolor) / 255 ) * 72  ; 'item 1' code only uses the green portion of rgb
       set depth depth-in-ft * 0.3048
       if current-depth > 0 and depth = 0
       [ set detritus detritus + DOC + POC + phyto + macro + waterdecomp + seddecomp + herbivore + sedconsumer + consum
         set DOC 0
         set POC 0
         set phyto 0
         set macro 0
         set waterdecomp 0
         set seddecomp 0
         set herbivore 0
         set sedconsumer 0
         set consum 0 
       ]
       
       if current-depth = 0 and depth > 0
       [ set detritus detritus * 0.50 ]
       
       set current-depth ( item 1 pcolor / 255 ) * 72 * 0.3048 ; grabs the current depth before the map update in the next time step, so it can be compared to the new depth 
     ]
     import-pcolors-rgb v-maps
     ask patches with  [pcolor != 0] 
     [ 
       set velocity-in-f/s ( (255 - last pcolor ) / 255 ) * 13  ; 'last' code only uses the blue portion of rgb
       set velocity velocity-in-f/s * 0.3048 * max-timestep 
       set velpoc velocity-in-f/s * 0.3048
     ]
   
     file-open vct-maps
     while [ not file-at-end? ]
     [ 
       ask patch file-read file-read
       [ set px-vector file-read * 0.3048 * max-timestep
         set py-vector file-read * 0.3048 * max-timestep
       ]
     ]
     file-close
end

;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
; function - this is the hydrological flow routine, which runs at a different timestep
; *need patch selection - currently random - should this be algorithmic? 
to update-patches  
  ;ifelse ticks mod 24 < 14 [ set surface-light 500] [ set surface-light .01] 
  ; set surface-light 500 * sin(2 * pi* ticks / 24)
    ; this appears to be a daylight routine - *need details 
  set surface-light par
  ; set day round (ticks / 1440)
  set turbidity ( .29587 + (.023757 * TSS) + (k-macro * macro) + (k-phyto + phyto) + (k-herbivore * herbivore) + (k-POC * POC))
  set  bottom-light (surface-light * exp ( (- depth) *  turbidity ))
    ; Environmetrics p3 
  set Aj-peri macro / 10 
  set Gj-peri macro / 2
  set Aj-seddecomp detritus / 20 
  set Gj-seddecomp detritus / 5
    ; *need references and definitions
end

to update-environmentals
  if ( ticks / 24 ) = ( int ( ticks / 24 )); updates daily (ticks = 1 was used to trick coding to skip 1st line in file)
  [
     update-temp     
     choose-hydro-map
     if hydro-changed = 1  [update-hydro-map]
  ]
  update-par
end  

to update-par
  file-open "data/par.txt"; hourly time-scale
  set par file-read
end

to update-temp
  file-open "data/water-temp.txt"; daily time-scale
  set temp file-read
end

to go-macro
  let base-temp 19.7 ; base temp for nominal growth
  set Q10 theta ^ (temp - base-temp)
  ifelse velpoc < 1 
    [set K  274 - (274 * velpoc)]  
    [set K 0.01]
    ; *need reference
  let macro-turbidity ( .29587 + (.023757 * TSS)  + (k-phyto + phyto) + (k-herbivore * herbivore) + (k-POC * POC))
    ; *need reference - if this equation repeats, let's make an explicit function
  let macro-light (surface-light * exp ( (- depth) *  macro-turbidity ))
    ; *need reference, though this looks familiar
  set gross-photo-macro (.08 * macro * ( macro-light / (macro-light + 10)) * Q10 * (K - macro) / K )
    ; .08 estimated from Kaj Sand-Jensen fig 7
  set respiration-macro .04 / 24 * macro * Q10
    ; *need reference
  set senescence-macro .05 / 24  * macro  
    ; *need reference
  ; values range between .05 and 3m a Garby et al 2006 - what is this referring to?
  ; set scouring-macro scouring * macro
  set growth-macro gross-photo-macro - respiration-macro - senescence-macro - scouring-macro
    ; summing above calcs *need to know about scouring discrepency
  set macro macro + growth-macro
  if macro < 0 [ set macro .001 ]
    ; *need reference - assuming you are just setting to prevent division by 0 elsewhere
end

to go-phyto
  let base-temp 8 ; base temp for nominal growth
  set Q10 theta ^ (temp - base-temp)
  let km 10  ;half saturation constant   
  set Q10 theta ^ (temp - base-temp)
  set light-k .4 ; this is the attenuation coefficient of phytoplank m^2/g of phyto plankton. 
  set respiration-phyto .1 / 24 * phyto * Q10
  set gross-photo-phyto ( (1.96 / 14) / k-phyto ) * ln (( km + surface-light) / (km + surface-light * e ^ ( -1 * phyto * K-phyto * depth ))) * ( 1 / depth) * ((phyto * K-phyto)/ turbidity) * Q10
  set excretion-phyto .05 / 24 * phyto
  set senescence-phyto .02 / 24  * phyto  ; values range between .05 and 3
  set growth-phyto gross-photo-phyto - excretion-phyto - respiration-phyto - senescence-phyto
end

; function - primary herbivore routine
to go-herbivore
  set herbivore-phyto-prey-limitation phyto / (ai-herbivore-phyto - gi-herbivore-phyto)
  if herbivore-phyto-prey-limitation > 1 [set herbivore-phyto-prey-limitation 1]
  if herbivore-phyto-prey-limitation < 0 [set herbivore-phyto-prey-limitation 0]
  set herbivore-peri-prey-limitation  peri / (ai-herbivore-peri - gi-herbivore-peri)
  if herbivore-peri-prey-limitation > 1 [set herbivore-peri-prey-limitation 1]
  if herbivore-peri-prey-limitation < 0 [set herbivore-peri-prey-limitation 0]
  set herbivore-waterdecomp-prey-limitation  waterdecomp / (ai-herbivore-waterdecomp - gi-herbivore-waterdecomp)
  if herbivore-waterdecomp-prey-limitation > 1 [set herbivore-waterdecomp-prey-limitation 1]
  if herbivore-waterdecomp-prey-limitation < 0 [set herbivore-waterdecomp-prey-limitation 0]
  set herbivore-space-limitation 1 - ((herbivore - Aj-herbivore) / (Gj-herbivore - Aj-herbivore))
  if herbivore-space-limitation > 1 [set herbivore-space-limitation 1]
  if herbivore-space-limitation < 0 [set herbivore-space-limitation 0]
      ; *need references
  set herbivore-pred-phyto (pref-herbivore-phyto * max-herbivore * herbivore * herbivore-space-limitation * herbivore-phyto-prey-limitation)
      ; *need reference
  set herbivore-ingest-phyto herbivore-pred-phyto * (1 - herbivore-egestion)
      ; *need reference
  set herbivore-pred-peri (pref-herbivore-peri * max-herbivore * herbivore * herbivore-space-limitation * herbivore-peri-prey-limitation)
      ; *need reference
  set herbivore-ingest-peri herbivore-pred-peri * (1 - herbivore-egestion)
      ; *need reference
  set herbivore-pred-waterdecomp (pref-herbivore-waterdecomp * max-herbivore * herbivore * herbivore-space-limitation * herbivore-waterdecomp-prey-limitation)
      ; *need reference
  set herbivore-ingest-waterdecomp herbivore-pred-waterdecomp * (1 - herbivore-egestion)
      ; *need reference
  set herbivore-respiration r-herbivore * herbivore 
      ; *need reference
  set herbivore-excretion e-herbivore * herbivore
      ; *need reference
  set herbivore-senescence s-herbivore * herbivore
      ; *need reference
  ;;set herbivore-phys-loss (herbivore-respiration + herbivore-excretion + herbivore-senescence)
      ; *need reference
  ; let theta 1.072 - temperature dependence...
  ; let base-temp 8 ; base temp for nominal growth
  ; set Q10 theta ^ (temp - base-temp)
end

; function - primary water decomposing routine
to go-waterdecomp
  set waterdecomp-doc-prey-limitation  doc / (ai-waterdecomp-doc - gi-waterdecomp-doc)
  if waterdecomp-doc-prey-limitation > 1 [set waterdecomp-doc-prey-limitation 1]
  if waterdecomp-doc-prey-limitation < 0 [set waterdecomp-doc-prey-limitation 0]
  set waterdecomp-poc-prey-limitation  poc / (ai-waterdecomp-poc - gi-waterdecomp-poc)
  if waterdecomp-poc-prey-limitation > 1 [set waterdecomp-poc-prey-limitation 1]
  if waterdecomp-poc-prey-limitation < 0 [set waterdecomp-poc-prey-limitation 0]
  set waterdecomp-space-limitation 1 - ((waterdecomp - Aj-waterdecomp) / (Gj-waterdecomp - Aj-waterdecomp))
  if waterdecomp-space-limitation > 1 [set waterdecomp-space-limitation 1]
  if waterdecomp-space-limitation < 0 [set waterdecomp-space-limitation 0]
      ; *need references
    ; PIij * THETAj * Xj * Fij * Fjj
  set waterdecomp-pred-doc (pref-waterdecomp-doc * max-waterdecomp * waterdecomp * waterdecomp-space-limitation * waterdecomp-doc-prey-limitation)
  set waterdecomp-ingest-doc waterdecomp-pred-doc 
  set waterdecomp-pred-poc (pref-waterdecomp-poc * max-waterdecomp * waterdecomp * waterdecomp-space-limitation * waterdecomp-poc-prey-limitation)
  set waterdecomp-ingest-poc waterdecomp-pred-poc
      ; *need reference - we look to be redundant in this call
  set waterdecomp-respiration r-waterdecomp * waterdecomp 
  set waterdecomp-excretion e-waterdecomp * waterdecomp
  set waterdecomp-senescence s-waterdecomp  * waterdecomp 
  ;;set waterdecomp-phys-loss (waterdecomp-respiration + waterdecomp-excretion + waterdecomp-senescence)
      ; sum above variables
end

; function - primary sedentary decomposing routine
to go-seddecomp
  set seddecomp-detritus-prey-limitation  detritus / (ai-seddecomp-detritus - gi-seddecomp-detritus)
  if seddecomp-detritus-prey-limitation > 1 [set seddecomp-detritus-prey-limitation 1]
  if seddecomp-detritus-prey-limitation < 0 [set seddecomp-detritus-prey-limitation 0]
      ; *need references
  set seddecomp-space-limitation 1 - ((seddecomp - Aj-seddecomp) / (Gj-seddecomp - Aj-seddecomp))
  if seddecomp-space-limitation > 1 [set seddecomp-space-limitation 1]
  if seddecomp-space-limitation < 0 [set seddecomp-space-limitation 0]
      ; *need references
  set seddecomp-pred-detritus (pref-seddecomp-detritus * max-seddecomp * seddecomp * seddecomp-detritus-prey-limitation * seddecomp-space-limitation)
      ; *need references
  set seddecomp-ingest-detritus seddecomp-pred-detritus
      ; *need references 
  set seddecomp-respiration r-seddecomp * seddecomp
      ; *need references 
  set seddecomp-excretion e-seddecomp * seddecomp
      ; *need references
  set seddecomp-senescence s-seddecomp * seddecomp 
      ; *need references
end


; function - primary sedentary consuming routine
to go-sedconsumer
  set sedconsumer-seddecomp-prey-limitation  seddecomp / (ai-sedconsumer-seddecomp - gi-sedconsumer-seddecomp)
  if sedconsumer-seddecomp-prey-limitation > 1 [set sedconsumer-seddecomp-prey-limitation 1]
  if sedconsumer-seddecomp-prey-limitation < 0 [set sedconsumer-seddecomp-prey-limitation 0]
      ; *need references
  set sedconsumer-peri-prey-limitation  peri / (ai-sedconsumer-peri - gi-sedconsumer-peri)
  if sedconsumer-peri-prey-limitation > 1 [set sedconsumer-peri-prey-limitation 1]
  if sedconsumer-peri-prey-limitation < 0 [set sedconsumer-peri-prey-limitation 0]
      ; *need references
  set sedconsumer-detritus-prey-limitation  detritus / (ai-sedconsumer-detritus - gi-sedconsumer-detritus)
  if sedconsumer-detritus-prey-limitation > 1 [set sedconsumer-detritus-prey-limitation 1]
  if sedconsumer-detritus-prey-limitation < 0 [set sedconsumer-detritus-prey-limitation 0]
      ; *need references
  set sedconsumer-space-limitation 1 - ((sedconsumer - Aj-sedconsumer) / (Gj-sedconsumer - Aj-sedconsumer))
  if sedconsumer-space-limitation > 1 [set sedconsumer-space-limitation 1]
  if sedconsumer-space-limitation < 0 [set sedconsumer-space-limitation 0]
      ; *need references
  set sedconsumer-pred-peri (pref-sedconsumer-peri * max-sedconsumer * sedconsumer * sedconsumer-space-limitation * sedconsumer-peri-prey-limitation)
      ; *need references
  set sedconsumer-ingest-peri sedconsumer-pred-peri * (1 - sedconsumer-egestion-seddecomp)
      ; *need references
  set sedconsumer-pred-seddecomp (pref-sedconsumer-seddecomp * max-sedconsumer * sedconsumer * sedconsumer-space-limitation * sedconsumer-seddecomp-prey-limitation)
      ; *need references
  set sedconsumer-ingest-seddecomp sedconsumer-pred-seddecomp * (1 - sedconsumer-egestion-seddecomp)
      ; *need references
  set sedconsumer-pred-detritus (pref-sedconsumer-detritus * max-sedconsumer * sedconsumer * sedconsumer-space-limitation * sedconsumer-detritus-prey-limitation)
      ; *need references
  set sedconsumer-ingest-detritus sedconsumer-pred-detritus * (1 - sedconsumer-egestion-detritus)
      ; *need references
  set sedconsumer-respiration r-sedconsumer * sedconsumer
      ; *need references
  set sedconsumer-excretion e-sedconsumer * sedconsumer
      ; *need references
  set sedconsumer-senescence s-sedconsumer * sedconsumer 
      ; *need references
  ;;set sedconsumer-phys-loss (sedconsumer-respiration + sedconsumer-excretion + sedconsumer-senescence)
      ; sums up all calculations
end

; function - primary consuming routine
to go-consum
  set consum-sedconsumer-prey-limitation  sedconsumer / (ai-consum-sedconsumer - gi-consum-sedconsumer)
  if consum-sedconsumer-prey-limitation > 1 [set consum-sedconsumer-prey-limitation 1]
  if consum-sedconsumer-prey-limitation < 0 [set consum-sedconsumer-prey-limitation 0]
      ; *need references
  set consum-herbivore-prey-limitation  herbivore / (ai-consum-herbivore - gi-consum-herbivore)
  if consum-herbivore-prey-limitation > 1 [set consum-herbivore-prey-limitation 1]
  if consum-herbivore-prey-limitation < 0 [set consum-herbivore-prey-limitation 0]
      ; *need references
  set consum-space-limitation 1 - ((consum - Aj-consum) / (Gj-consum - Aj-consum))
  if consum-space-limitation > 1 [set consum-space-limitation 1]
  if consum-space-limitation < 0 [set consum-space-limitation 0]
      ; *need references
  set consum-pred-herbivore (pref-consum-herbivore * max-consum * consum * consum-space-limitation * consum-herbivore-prey-limitation)
      ; *need references
  set consum-ingest-herbivore consum-pred-herbivore * (1 - consum-egestion)
      ; *need references
  set consum-pred-sedconsumer (pref-consum-sedconsumer * max-consum * consum * consum-space-limitation * consum-sedconsumer-prey-limitation)
      ; *need references
  set consum-ingest-sedconsumer consum-pred-sedconsumer * (1 - consum-egestion)
      ; *need references
  set consum-respiration r-consum * consum 
      ; *need references
  set consum-excretion e-consum * consum
      ; *need references
  set consum-senescence s-consum  * consum 
      ; *need references
  ;;set consum-phys-loss (consum-respiration + consum-excretion + consum-senescence)
      ; sums up all calculations
end

; function - dissolved organic carbon routine
to go-DOC
  set macro-exudation .04 * macro 
      ;Wetzel lit.
  set micro-death senescence-macro * .01 + senescence-phyto * .01 
      ;Dead objects < 1um are considered DOC in Wetzel book..
  set excretion herbivore-excretion + waterdecomp-excretion + seddecomp-excretion + sedconsumer-excretion + consum-excretion + excretion-phyto + peri-excretion
      ; *need reference
  set flocculation .01 * DOC
      ; *need reference
  set DOC-growth macro-exudation + micro-death + excretion
      ; sums calculations into variable
end

; function - particulate organic carbon routine
to go-POC
  if velocity > 0
    [ set detritus-POC-transfer detritus * (.25 * log (((velocity / 40) + .0001) + 1) 10 )]
       ; exchange between POC and detritus determined by an approximation of Stoke's Law
  if detritus-POC-transfer > 1 
    [set detritus-POC-transfer 1]
        ; *need reference
  set small-death senescence-macro * .09 + senescence-phyto * .09 + scouring-macro * 0.1 + waterdecomp-senescence * .7 + .3 * peri-senescence ;Dead objects < 10um will be in water column in Wetzel book.
      ; *need reference
  set POC-growth flocculation + detritus-POC-transfer
      ; flocculation, leaching approximation
end 

; function - detritus routine
to go-detritus
  if velocity > 0
    [ set POC-detritus-transfer POC * ( 1 - (.25 * log (((velpoc / 40) + .0001) + 1) 10 )) ]
      ; *need reference
  if POC-detritus-transfer < 0
    [set POC-detritus-transfer 0]
      ; *need reference
  if velocity = 0
    [ set POC-detritus-transfer POC * .9 ]
      ; *need reference
  if gross-photo-macro < 0
   [ set macro-death 0 - gross-photo-macro]
     ; to incorporate scoured macrophytes into detritus
  set large-death senescence-macro * .9 + scouring-macro * 0.9 + senescence-phyto * .9 + seddecomp-senescence + waterdecomp-senescence * .3 +
    herbivore-senescence + sedconsumer-senescence + consum-senescence + .07 * peri-senescence
      ; Dead objects > 10um will sink Wetzel book..
  set egestion herbivore-egestion + sedconsumer-egestion + consum-egestion
  set detritus-growth large-death + POC-detritus-transfer + egestion + macro-death
      ; *need reference
  ;show detritus-growth   
end

; function - detritus routine
to pred-phyto
  set phyto phyto + growth-phyto - herbivore-pred-phyto
  if phyto < 0 [ set phyto .001 ]
end

; function - detritus routine
to pred-herbivore
  set herbivore herbivore + herbivore-ingest-phyto + herbivore-ingest-peri + herbivore-ingest-waterdecomp - (herbivore-respiration + herbivore-excretion + herbivore-senescence) - consum-pred-herbivore
  if herbivore < 0 [ set herbivore 0.001 ]
end

; function - detritus routine
to pred-seddecomp
  set seddecomp seddecomp + seddecomp-ingest-detritus - (seddecomp-respiration + seddecomp-excretion + seddecomp-senescence) - sedconsumer-pred-seddecomp
  if seddecomp < 0 [ set seddecomp .001 ]
end

; function - detritus routine
to pred-waterdecomp
  set waterdecomp waterdecomp + waterdecomp-ingest-doc + waterdecomp-ingest-poc - (waterdecomp-respiration + waterdecomp-excretion + waterdecomp-senescence) - herbivore-pred-waterdecomp
  if waterdecomp < 0 [ set waterdecomp .001 ]
end

; function - detritus routine
to pred-sedconsumer
  set sedconsumer sedconsumer + sedconsumer-ingest-peri + sedconsumer-ingest-seddecomp - (sedconsumer-respiration + sedconsumer-excretion + sedconsumer-senescence) - consum-pred-sedconsumer
  if sedconsumer < 0 [set sedconsumer .001 ]
end

; function - detritus routine
to pred-detritus
   set detritus-POC-transfer detritus * ((.25 * log ((velpoc / 40) + .01) 10) + .5) 
  set detritus detritus + detritus-growth - seddecomp-pred-detritus - detritus-POC-transfer
  if detritus < 0 [ set detritus .001 ]
end

; function - detritus routine
to pred-DOC 
  set DOC DOC + DOC-growth - waterdecomp-pred-doc - flocculation
  if DOC < 0 [ set DOC .001 ]
end

; function - detritus routine
to pred-POC
  set POC POC + POC-growth - waterdecomp-pred-poc - POC-detritus-transfer
  if POC < 0 [ set POC .001 ]
end

; function - detritus routine
to pred-consum 
  set consum consum + consum-ingest-herbivore + consum-ingest-sedconsumer -  (consum-respiration + consum-excretion + consum-senescence) 
  if consum < 0 [set consum .001]
end


to flow-carbon
;;; Using patch-at, we only need to know which direction the carbon flows and our choices are polarized
;;; either, -1/0/1, since we just assume some direction, we only test for +/- and move towards 3 patches
;;; if the vector is zero, nothing bad happens, we just spun our wheels a little      
;;; Since we are using cartesian corrdinates, we use the absolutes of the x and y vectors to equate the fraction of 
;;; the carbon to be moved - we have a top/bottom patch, corner patch, right/left patch
  set max-area patch-length ^ 2
  set cornerpatch abs( py-vector * px-vector  ) / max-area
  set tbpatch abs( py-vector * ( patch-length - abs(px-vector) ) ) / max-area
  set rlpatch abs( px-vector * ( patch-length - abs(py-vector) ) ) / max-area

  set origDOC DOC ; need to carry the carbon of the original patch to figure the result in the neighboring patches
  set origPOC POC
  set origPHYTO PHYTO
  set origWATERDECOMP WATERDECOMP
  set tbMvd 0 ; if a neighbor patch is dry, the carbon doesn't move  in that direction 
  set cMvd 0  ; this idea can be revisited to see if we want to move more to other patches
  set rlMvd 0
  
  set px 0  ; this little trick checks the polarity of the vector to allow us to select the correct 'patch-at'
  set py 0
  ifelse px-vector >= 0 [set px 1] [set px -1]
  ifelse py-vector >= 0 [set py 1] [set py -1]

  ; since we had to process our own vector data, we do not have to worry about non-existent patches
  ; plus I turned wrapping back on - there are not any patches that touch across the world borders
  
  ask patch-at 0 py [
    if depth > 0 [
      set DOC (DOC + origDOC * tbpatch)
      set POC (POC + origPOC * tbpatch)
      set PHYTO (PHYTO + origPHYTO * tbpatch)
      set WATERDECOMP (WATERDECOMP + origWATERDECOMP * tbpatch)
      set tbMvd 1
    ]
  ]

  ask patch-at px py [
    if depth > 0 [
      set DOC (DOC + origDOC * cornerpatch)
      set POC (POC + origPOC * cornerpatch)
      set PHYTO (PHYTO + origPHYTO * cornerpatch)
      set WATERDECOMP (WATERDECOMP + origWATERDECOMP * cornerpatch)            
      set cMvd 1
    ]
  ]

  ask  patch-at px 0 [
    if depth > 0 [
      set DOC (DOC + origDOC * rlpatch)
      set POC (POC + origPOC * rlpatch)
      set PHYTO (PHYTO + origPHYTO * rlpatch)
      set WATERDECOMP (WATERDECOMP + origWATERDECOMP * rlpatch)            
      set rlMvd 1
    ]
  ]
  
  ; if a patch was dry and the carbon didn't move, we zero out the loss in the original patch
  let patch-loss (tbpatch * tbMvd + cornerpatch * cMvd + rlpatch * rlMvd)  
  set DOC (DOC - DOC * patch-loss)
  set POC (POC - POC * patch-loss)
  set PHYTO (PHYTO - PHYTO * patch-loss)
  set WATERDECOMP (WATERDECOMP - WATERDECOMP * patch-loss)                
end

to export-fc
  file-open "data/export-fs.txt"
  foreach sort patches with [depth > 0 and output != 1] [
  ask ? 
  [ file-print (word pxcor " " pycor " " DOC " " POC " " PHYTO " " WATERDECOMP)]]
  file-close
end

to update-color
  let patches-to-display patches with [ depth > 0 and output != 1 and pycor > 100 ]
  let c red
  
  if which-stock? = "macro"        [set c green   ask patches-to-display [set tmp macro ]]
  if which-stock? = "phyto"        [set c green   ask patches-to-display [set tmp phyto ]]
  if which-stock? = "waterdecomp"  [set c green   ask patches-to-display [set tmp waterdecomp ]]
  if which-stock? = "seddecomp"    [set c sky     ask patches-to-display [set tmp seddecomp ]]
  if which-stock? = "herbivore"    [set c violet  ask patches-to-display [set tmp herbivore ]]
  if which-stock? = "consum"       [set c orange  ask patches-to-display [set tmp consum ]]
  if which-stock? = "DOC"          [set c yellow  ask patches-to-display [set tmp DOC ]]
  if which-stock? = "POC"          [set c blue    ask patches-to-display [set tmp POC ]]
  if which-stock? = "detritus"     [set c brown   ask patches-to-display [set tmp detritus ]]
  if which-stock? = "sedconsumer"  [set c yellow  ask patches-to-display [set tmp sedconsumer ] ]
   
  ask patches-to-display [ if (tmp > 15) [ set tmp 15 ] ]
  let max-val max [ tmp ] of patches-to-display
   ask patches-to-display [ set pcolor ( c + 4.9 - ( ((tmp / max-val) * 10) ) ) ]
end


to do-plots
  let study-area-patches patches with [ depth > 0 ]
  set-current-plot "biomass"
  set-current-plot-pen "DOC"
  plot mean [DOC] of study-area-patches
  set-current-plot-pen "POC"
  plot mean [POC] of study-area-patches
  set-current-plot-pen "phyto"
  plot mean [phyto] of study-area-patches
  set-current-plot-pen "macro"
  plot mean [macro] of study-area-patches
  set-current-plot-pen "herbivore"
  plot mean [herbivore] of study-area-patches
  set-current-plot-pen "WD"
  plot mean [waterdecomp] of study-area-patches
  set-current-plot-pen "consum"
  plot mean [consum] of study-area-patches
  set-current-plot-pen "detritus"
  plot mean [detritus] of study-area-patches
  set-current-plot-pen "seddecomp"
  plot mean [seddecomp] of study-area-patches
  set-current-plot-pen "sedconsumer"
  plot mean [sedconsumer] of study-area-patches
  set-current-plot "output"
  set-current-plot-pen "Output DOC"
  plot sum [DOC] of patches with [pycor = 100]
  set-current-plot-pen "Output POC"
  plot sum [POC] of patches with [pycor = 100]
  set-current-plot-pen "Output Phytoplankton"
  plot sum [phyto] of patches with [pycor = 100]
  set-current-plot-pen "Output Waterdecomposers"
  plot sum [waterdecomp] of patches with [pycor = 100]
end
; for monitors of a variable of different depths: mean [variable] of patches with [ depth >= value and depth < value]

; function - carbon input routine
; adds 10g C per cell to four trophic levels at the river head 
to input
 ; will change in bigger map
  set DOC 10
  set POC 10
  set phyto 10
  set waterdecomp 10
end

;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;;; END FUNCTIONS in SETUP ROTUTINE
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;


;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;;; FUNCTIONS in Interface
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
 
to pred-peri
  set peri peri + peri-ingest-doc + peri-ingest-poc - peri-phys-loss - herbivore-pred-peri - sedconsumer-pred-peri
  if peri < 0 [ set peri .001 ]
end
@#$#@#$#@
GRAPHICS-WINDOW
152
10
750
954
-1
-1
1.5
1
10
1
1
1
0
1
1
1
0
391
0
608
1
1
1
ticks
30.0

BUTTON
3
10
66
43
NIL
setup
NIL
1
T
OBSERVER
NIL
NIL
NIL
NIL
1

BUTTON
75
10
138
43
NIL
go
T
1
T
OBSERVER
NIL
NIL
NIL
NIL
1

PLOT
751
235
1150
510
biomass
Hours
Mean Grams Per Cell
0.0
10.0
0.0
10.0
true
true
"" ""
PENS
"DOC" 1.0 0 -16777216 true "" ""
"POC" 1.0 0 -2674135 true "" ""
"phyto" 1.0 0 -11221820 true "" ""
"macro" 1.0 0 -10899396 true "" ""
"herbivore" 1.0 0 -8630108 true "" ""
"WD" 1.0 0 -13345367 true "" ""
"consum" 1.0 0 -6459832 true "" ""
"detritus" 1.0 0 -14835848 true "" ""
"seddecomp" 1.0 0 -2064490 true "" ""
"sedconsumer" 1.0 0 -955883 true "" ""

CHOOSER
1
343
139
388
which-stock?
which-stock?
"macro" "phyto" "waterdecomp" "seddecomp" "herbivore" "consum" "DOC" "POC" "detritus" "sedconsumer"
6

SLIDER
2
101
146
134
TSS
TSS
0
1
0.72
.01
1
NIL
HORIZONTAL

MONITOR
751
72
809
117
NIL
day
17
1
11

BUTTON
5
201
106
235
inject DOC
ask patch 107 168 [set DOC DOC + 500000]\n\n
NIL
1
T
OBSERVER
NIL
NIL
NIL
NIL
1

PLOT
751
521
1220
742
output
Hours
Mass
0.0
10.0
0.0
10.0
true
true
"" ""
PENS
"Output DOC" 1.0 0 -16777216 true "" ""
"Output POC" 1.0 0 -5825686 true "" ""
"Output Phytoplankton" 1.0 0 -14835848 true "" ""
"Output Waterdecomposers" 1.0 0 -8630108 true "" ""

BUTTON
1
417
137
451
Update Map
update-color
NIL
1
T
OBSERVER
NIL
NIL
NIL
NIL
1

MONITOR
812
73
869
118
hour
hour
2
1
11

TEXTBOX
4
135
154
153
Set Total Suspended Solids
11
0.0
1

TEXTBOX
6
237
156
307
Increase Dissolved\nOrganic Carbon\nby 500000 at  \npatch 107 168 \n\n
11
0.0
1

TEXTBOX
4
392
154
410
Choose map to display
11
0.0
1

@#$#@#$#@
## PURPOSE

This model combines ecological and hydrological processes to simulate carbon flow through Pool 5 of the Upper Mississippi River, a 24 km section extending from Navigation Dam No. 4 at Alma, Wisconsin, downstream to Navigation Dam No. 5 near Whitman, Minnesota. A basic carbon cycle is modeled using simple food web interaction. Water depth, current velocity, and current direction were derived for Pool 5, assuming a moderate discharge of 88,000 cfs, using the Adaptive Hydraulics two-dimensional hydraulic simulation model (USACE 2008) to describe the spatial distribution of dissolved carbon.

## STATE VARIABLES AND SCALES

Water depth, velocity, and flow direction in Pool 5, as reported by the US Army Corps of Engineers, were plotted on digital maps using a color gradient. These maps were divided into patches of 30 x 30 m (Figure 9-2), and the carbon cycle was modeled within each patch (as described in the �Submodels� section). This carbon cycle model, originally developed in 1983 for Pool 19 of the Upper Mississippi River (Sparks 1985), simulates the transfer of labile carbon among trophic levels within each patch. It includes eleven basic carbon states: dissolved organic carbon (DOC), particulate organic carbon (POC), detritus, phytoplankton, macrophytes, periphyton, herbivores, decomposers in the water column, decomposers in the sediment, consumers in the water column, and consumers in the sediments. The abundance of each of these stocks was expressed in grams of carbon per patch.  
While there is evidence that primary productivity in large river ecosystems is influenced by nitrogen, phosphorous, and other nutrients (Allan and Castillo 2007; Francoeur 2001; Houser and Richardson 2010), there is other evidence that the Upper Mississippi River is not nutrient-limited and that primary production, especially in the main channel, is often limited by factors such as light availability and temperature (Houser et al. 2010; Huff 1986; Owens and Crumpton 1995). For these reasons, and to maintain simplicity, the effects of nitrogen and phosphorous are not simulated in this model. Primary productivity is based on temperature, light, and water velocity. Transfer of carbon from one stock to another is based on biomass-dependent interactions between stock. The movement of labile carbon between patches is modeled on the basis of water velocity and flow direction (see �Incorporation of Hydraulics� section). The time step for the model is set at 1 hour for the purpose of corresponding with many of the measurements reported for river processes in the literature and with the hydrology data.

## PROCESS OVERVIEW AND SCHEDULING

Within each patch, during each time step, carbon is first added to both autotrophic and heterotrophic carbon stocks as dictated by the formulas for primary production and consumption. Then the carbon that was consumed by the heterotrophs is removed from the appropriate carbon stocks indicated by the food web. Finally, each stock is assigned a new biomass value that takes growth and predation into account, and the hydrologic component moves carbon among patches. Interactions between stocks are described below.

## INITIALIZATION

All carbon stocks transported in the current (phytoplankton, water column decomposers, DOC, POC) were arbitrarily set at 10 g of carbon per patch to start the model. These stocks also received influxes of 10 g of carbon in each of the patches bordering the upstream end of Pool 5 during each time step to simulate the influence of upstream carbon on the system. This amount was chosen based on organic suspended sediments data for Pool 4 from the Long-Term Resource Monitoring Program implemented by the US Geological Survey (http://www.umesc.usgs.gov/ltrmp.html). All other stocks were set at 1 g of carbon per cell to start the model and were not subject to influxes or outfluxes during the simulation.

## INPUT

The model takes input from GIS maps of water depth, velocity, and flow direction in Pool 5 of the Mississippi River, which were created from a 2-dimensional hydraulic model developed by the US Army Corps of Engineers (USACE 2008). The hydraulic model simulated a discharge of 2,097 m3sec-1. Although this discharge level is not common for periods longer than 1 month, it was chosen because the data from the hydraulic model were the most readily available source. Maps were divided into patches across a color gradient so that an average depth, velocity, and flow vector could be calculated for each patch. 

## AUTOTROPHIC STOCKS SUBMODEL

The two autotrophs considered for this model are macrophytes and phytoplankton. Macrophytes were modeled only in patches where water velocity was less than 1 m/s because higher velocities result in complete scouring.  
We modeled all autotrophic stocks using the general equation:  
....... dXj/dt = Prodj - Physj - Predj		Eq 1  
where the biomass of stock j equals the difference between primary production and physiological and predatory losses.  
Primary production for macrophytes is described using the maximum gross photosynthesis equation (Chambers et al. 1991; Sand-Jensen et al. 2007):  
....... Prodj = 0.08 * Xj * (l/(l+10)) * (K-Xj)/K		Eq 2  
where Xj is macrophyte biomass, K is the half-saturation constant, and l is the amount of light that reaches the macrophytes depending on the amount of surface light, depth, and total suspended solids:  
....... l = (s * e^ ( (-d) * b))		Eq 3  
where s is surface light, d is depth, and b is turbidity. Primary production for phytoplankton was modeled using a formula similar to that used for macrophytes (Huisman and Weissing 1994).  
Physiological losses of an autotrophic stock were modeled as:  
....... Physj = (�j + ?j + ?j) * Xj		Eq 4  
where �j is the specific physiological mortality rate, ?j is carbon lost to excretion, and ?j is carbon lost to respiration (Garbey et al. 2006). Carbon lost via physiological mortality is transferred to the detritus, POC, and DOC stocks. Carbon lost via excretion enters the DOC stock. Carbon lost to respiration by a stock is removed from the system. In this version of the model, the phytoplankton stock--not the macrophyte stock--loses carbon to herbivores. These losses are described in the equations for heterotrophic stocks.

## HETEROTROPHIC STOCKS

Heterotrophic stocks included aufwuchs (microbial component), herbivores, consumers, and decomposers. We modeled all of these stocks using the general equations developed by Wiegert (1975). Biomass of a stock at time t + 1 is determined by the formula:  
......... dXj/dt = Cj * (1 - ?j) - Physj - Predj		Eq 5  
where the biomass of stock j equals the difference between consumption corrected for egestion (?j) and physiological and predatory losses. Egested material enters the detrital food web as POC, much of which may be converted to detritus depending on the hydrological conditions of the cell. Consumption (C) by stock j is determined by the summation of consumption of each available prey type (m) of predator j, given by the formula:  
......... Cj = � (�ij * ?j * Xj * fij * fjj) (� is sum from i = 1 to m)		Eq 6  
where �ij is the preference of stock j for prey stock i, ?j is the maximum rate of consumption by predator stock j, and Xj is the biomass of predator stock j. The functional response of predator stocks are determined by fij and fjj, which represent prey and space limitation functions:  
......... fij = y / (Aixy - Gixy)		Eq 7  
......... fjj = 1 - ((x � Ajx) / (Gjx - Ajx))		Eq 8  
where x is the present biomass of the heterotrophic stock in question and y is the present biomass of the prey stock being fed on by the heterotrophic stock. Because population growth rates can be limited by both prey availability and intraspecific interference at high population densities, both fij and fjj produce values between 0 and 1. At high prey densities, predators will not be resource-limited and thus feed at the maximum rate (?j). However, below a prey density threshold (Aij), consumption will be reduced and will eventually fall to zero when prey densities are below a refuge level (Gij). If a predator stock feeds only on one prey type, then �ij is set to 1. However, when a predator stock feeds on multiple stocks, �ij is a function of innate prey preference (Pij) for specific stocks and the biomass of those stocks in the cell, given as:  
......... �ij = (Pij * fij)/� (Pij * fij) (� is sum from i =1 to m)		Eq 9  
For simplicity we assume fij decreases linearly between Aij and Gij. Space limitation is determined similarly, except values of Ajj and Gjj relate to predator density.  
Stock-specific values for Aij, Ajj , Gij, Gjj, and Pij were adapted from Pace (1984) and are listed in Tables 9-1 and 9-2. However, these values are likely to be system-specific and thus should be calibrated with independent data from the system to be modeled. Calibration for this system will occur with further model validation and sensitivity analyses.  
Physiological losses of a heterotrophic stock were modeled as above for autotrophic stocks:  
......... Physj = (�j + ?j + ?j) * Xj		Eq 10  
Carbon lost to predation is determined by the summation of losses to each predator stock (k).  
......... Predj = � (�jk * ?k * Xk * fjk * fkk) (� is sum from k = 1 to n)		Eq 11

## DETRITUS, POC, AND DOC

Both the detritus and POC carbon stocks acquire carbon from dead organic matter that is greater than 1?m in size. However, POC refers to matter in the water column and detritus refers to matter in the sediments. For this model, senesced material was deemed detritus if particle size was greater than 10?m (Wetzel 2001). The percent of senesced material transferred to the detritus stock varied according to its source (Table 9-3). All egested carbon contributes to detritus (Table 9-3). Carbon lost to POC from the detritus category is described as:   
......... Transj = Xj * (2.5 log ((v / 40.0) + 0.0001) + 0.5)		Eq 12  
where v is equal to the water velocity, and higher velocities result in a greater proportion of organic matter in the water column. Similarly, the transfer of carbon from the POC stock to the detritus stock is described as the product of POC biomass and (1 � Transj). When velocity is equal to zero, 90% of the carbon in the POC stock settles out and becomes detritus.  
All senesced carbon in the model that is not assigned to the detritus stock is assigned to the POC stock, with the exception of a small percentage that is assigned to the DOC stock (Table 9-3). Carbon is also transferred to POC from DOC at a rate of 1% due to flocculation.  
The DOC carbon stock acquires carbon from dead or excreted objects that measure less than 1 �m in size (Wetzel 2001). In this model, in addition to some senesced carbon, all carbon excreted or released as exudates contributes to DOC, seen in Table 9-3 (Wetzel 1984).

## INCORPORATION OF HYDRAULICS

Mobile carbon from each cell was modeled to flow at the same velocity as water, and the amount of carbon transferred between cells was calculated using trigonometric functions based on the speed and direction of water flow in each cell. Since most fish actively move to find food, consumers in the water column in this model were not affected by hydraulics.
@#$#@#$#@
default
true
0
Polygon -7500403 true true 150 5 40 250 150 205 260 250

airplane
true
0
Polygon -7500403 true true 150 0 135 15 120 60 120 105 15 165 15 195 120 180 135 240 105 270 120 285 150 270 180 285 210 270 165 240 180 180 285 195 285 165 180 105 180 60 165 15

arrow
true
0
Polygon -7500403 true true 150 0 0 150 105 150 105 293 195 293 195 150 300 150

box
false
0
Polygon -7500403 true true 150 285 285 225 285 75 150 135
Polygon -7500403 true true 150 135 15 75 150 15 285 75
Polygon -7500403 true true 15 75 15 225 150 285 150 135
Line -16777216 false 150 285 150 135
Line -16777216 false 150 135 15 75
Line -16777216 false 150 135 285 75

bug
true
0
Circle -7500403 true true 96 182 108
Circle -7500403 true true 110 127 80
Circle -7500403 true true 110 75 80
Line -7500403 true 150 100 80 30
Line -7500403 true 150 100 220 30

butterfly
true
0
Polygon -7500403 true true 150 165 209 199 225 225 225 255 195 270 165 255 150 240
Polygon -7500403 true true 150 165 89 198 75 225 75 255 105 270 135 255 150 240
Polygon -7500403 true true 139 148 100 105 55 90 25 90 10 105 10 135 25 180 40 195 85 194 139 163
Polygon -7500403 true true 162 150 200 105 245 90 275 90 290 105 290 135 275 180 260 195 215 195 162 165
Polygon -16777216 true false 150 255 135 225 120 150 135 120 150 105 165 120 180 150 165 225
Circle -16777216 true false 135 90 30
Line -16777216 false 150 105 195 60
Line -16777216 false 150 105 105 60

car
false
0
Polygon -7500403 true true 300 180 279 164 261 144 240 135 226 132 213 106 203 84 185 63 159 50 135 50 75 60 0 150 0 165 0 225 300 225 300 180
Circle -16777216 true false 180 180 90
Circle -16777216 true false 30 180 90
Polygon -16777216 true false 162 80 132 78 134 135 209 135 194 105 189 96 180 89
Circle -7500403 true true 47 195 58
Circle -7500403 true true 195 195 58

circle
false
0
Circle -7500403 true true 0 0 300

circle 2
false
0
Circle -7500403 true true 0 0 300
Circle -16777216 true false 30 30 240

cow
false
0
Polygon -7500403 true true 200 193 197 249 179 249 177 196 166 187 140 189 93 191 78 179 72 211 49 209 48 181 37 149 25 120 25 89 45 72 103 84 179 75 198 76 252 64 272 81 293 103 285 121 255 121 242 118 224 167
Polygon -7500403 true true 73 210 86 251 62 249 48 208
Polygon -7500403 true true 25 114 16 195 9 204 23 213 25 200 39 123

cylinder
false
0
Circle -7500403 true true 0 0 300

dot
false
0
Circle -7500403 true true 90 90 120

face happy
false
0
Circle -7500403 true true 8 8 285
Circle -16777216 true false 60 75 60
Circle -16777216 true false 180 75 60
Polygon -16777216 true false 150 255 90 239 62 213 47 191 67 179 90 203 109 218 150 225 192 218 210 203 227 181 251 194 236 217 212 240

face neutral
false
0
Circle -7500403 true true 8 7 285
Circle -16777216 true false 60 75 60
Circle -16777216 true false 180 75 60
Rectangle -16777216 true false 60 195 240 225

face sad
false
0
Circle -7500403 true true 8 8 285
Circle -16777216 true false 60 75 60
Circle -16777216 true false 180 75 60
Polygon -16777216 true false 150 168 90 184 62 210 47 232 67 244 90 220 109 205 150 198 192 205 210 220 227 242 251 229 236 206 212 183

fish
false
0
Polygon -1 true false 44 131 21 87 15 86 0 120 15 150 0 180 13 214 20 212 45 166
Polygon -1 true false 135 195 119 235 95 218 76 210 46 204 60 165
Polygon -1 true false 75 45 83 77 71 103 86 114 166 78 135 60
Polygon -7500403 true true 30 136 151 77 226 81 280 119 292 146 292 160 287 170 270 195 195 210 151 212 30 166
Circle -16777216 true false 215 106 30

flag
false
0
Rectangle -7500403 true true 60 15 75 300
Polygon -7500403 true true 90 150 270 90 90 30
Line -7500403 true 75 135 90 135
Line -7500403 true 75 45 90 45

flower
false
0
Polygon -10899396 true false 135 120 165 165 180 210 180 240 150 300 165 300 195 240 195 195 165 135
Circle -7500403 true true 85 132 38
Circle -7500403 true true 130 147 38
Circle -7500403 true true 192 85 38
Circle -7500403 true true 85 40 38
Circle -7500403 true true 177 40 38
Circle -7500403 true true 177 132 38
Circle -7500403 true true 70 85 38
Circle -7500403 true true 130 25 38
Circle -7500403 true true 96 51 108
Circle -16777216 true false 113 68 74
Polygon -10899396 true false 189 233 219 188 249 173 279 188 234 218
Polygon -10899396 true false 180 255 150 210 105 210 75 240 135 240

house
false
0
Rectangle -7500403 true true 45 120 255 285
Rectangle -16777216 true false 120 210 180 285
Polygon -7500403 true true 15 120 150 15 285 120
Line -16777216 false 30 120 270 120

leaf
false
0
Polygon -7500403 true true 150 210 135 195 120 210 60 210 30 195 60 180 60 165 15 135 30 120 15 105 40 104 45 90 60 90 90 105 105 120 120 120 105 60 120 60 135 30 150 15 165 30 180 60 195 60 180 120 195 120 210 105 240 90 255 90 263 104 285 105 270 120 285 135 240 165 240 180 270 195 240 210 180 210 165 195
Polygon -7500403 true true 135 195 135 240 120 255 105 255 105 285 135 285 165 240 165 195

line
true
0
Line -7500403 true 150 0 150 300

line half
true
0
Line -7500403 true 150 0 150 150

pentagon
false
0
Polygon -7500403 true true 150 15 15 120 60 285 240 285 285 120

person
false
0
Circle -7500403 true true 110 5 80
Polygon -7500403 true true 105 90 120 195 90 285 105 300 135 300 150 225 165 300 195 300 210 285 180 195 195 90
Rectangle -7500403 true true 127 79 172 94
Polygon -7500403 true true 195 90 240 150 225 180 165 105
Polygon -7500403 true true 105 90 60 150 75 180 135 105

plant
false
0
Rectangle -7500403 true true 135 90 165 300
Polygon -7500403 true true 135 255 90 210 45 195 75 255 135 285
Polygon -7500403 true true 165 255 210 210 255 195 225 255 165 285
Polygon -7500403 true true 135 180 90 135 45 120 75 180 135 210
Polygon -7500403 true true 165 180 165 210 225 180 255 120 210 135
Polygon -7500403 true true 135 105 90 60 45 45 75 105 135 135
Polygon -7500403 true true 165 105 165 135 225 105 255 45 210 60
Polygon -7500403 true true 135 90 120 45 150 15 180 45 165 90

square
false
0
Rectangle -7500403 true true 30 30 270 270

square 2
false
0
Rectangle -7500403 true true 30 30 270 270
Rectangle -16777216 true false 60 60 240 240

star
false
0
Polygon -7500403 true true 151 1 185 108 298 108 207 175 242 282 151 216 59 282 94 175 3 108 116 108

target
false
0
Circle -7500403 true true 0 0 300
Circle -16777216 true false 30 30 240
Circle -7500403 true true 60 60 180
Circle -16777216 true false 90 90 120
Circle -7500403 true true 120 120 60

tree
false
0
Circle -7500403 true true 118 3 94
Rectangle -6459832 true false 120 195 180 300
Circle -7500403 true true 65 21 108
Circle -7500403 true true 116 41 127
Circle -7500403 true true 45 90 120
Circle -7500403 true true 104 74 152

triangle
false
0
Polygon -7500403 true true 150 30 15 255 285 255

triangle 2
false
0
Polygon -7500403 true true 150 30 15 255 285 255
Polygon -16777216 true false 151 99 225 223 75 224

truck
false
0
Rectangle -7500403 true true 4 45 195 187
Polygon -7500403 true true 296 193 296 150 259 134 244 104 208 104 207 194
Rectangle -1 true false 195 60 195 105
Polygon -16777216 true false 238 112 252 141 219 141 218 112
Circle -16777216 true false 234 174 42
Rectangle -7500403 true true 181 185 214 194
Circle -16777216 true false 144 174 42
Circle -16777216 true false 24 174 42
Circle -7500403 false true 24 174 42
Circle -7500403 false true 144 174 42
Circle -7500403 false true 234 174 42

turtle
true
0
Polygon -10899396 true false 215 204 240 233 246 254 228 266 215 252 193 210
Polygon -10899396 true false 195 90 225 75 245 75 260 89 269 108 261 124 240 105 225 105 210 105
Polygon -10899396 true false 105 90 75 75 55 75 40 89 31 108 39 124 60 105 75 105 90 105
Polygon -10899396 true false 132 85 134 64 107 51 108 17 150 2 192 18 192 52 169 65 172 87
Polygon -10899396 true false 85 204 60 233 54 254 72 266 85 252 107 210
Polygon -7500403 true true 119 75 179 75 209 101 224 135 220 225 175 261 128 261 81 224 74 135 88 99

wheel
false
0
Circle -7500403 true true 3 3 294
Circle -16777216 true false 30 30 240
Line -7500403 true 150 285 150 15
Line -7500403 true 15 150 285 150
Circle -7500403 true true 120 120 60
Line -7500403 true 216 40 79 269
Line -7500403 true 40 84 269 221
Line -7500403 true 40 216 269 79
Line -7500403 true 84 40 221 269

x
false
0
Polygon -7500403 true true 270 75 225 30 30 225 75 270
Polygon -7500403 true true 30 75 75 30 270 225 225 270

@#$#@#$#@
NetLogo 5.0RC3
@#$#@#$#@
@#$#@#$#@
@#$#@#$#@
<experiments>
  <experiment name="experiment" repetitions="1" runMetricsEveryStep="true">
    <setup>setup</setup>
    <go>go</go>
    <timeLimit steps="1"/>
    <metric>count turtles</metric>
    <enumeratedValueSet variable="which-stock?">
      <value value="&quot;macro&quot;"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="temp">
      <value value="17"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="TSS">
      <value value="0.72"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="patch-length">
      <value value="30"/>
    </enumeratedValueSet>
  </experiment>
</experiments>
@#$#@#$#@
@#$#@#$#@
default
0.0
-0.2 0 0.0 1.0
0.0 1 1.0 0.0
0.2 0 0.0 1.0
link direction
true
0
Line -7500403 true 150 150 90 180
Line -7500403 true 150 150 210 180

@#$#@#$#@
0
@#$#@#$#@
